var allowedVariants = ['normal', 'small-caps'];
/**
 * Gets font CSS shorthand property given element.
 *
 * @param {HTMLElement} element Element to get font CSS shorthand property from
 */
export function getFontShorthand(element) {
    if (!element) {
        return '';
    }
    var style = window.getComputedStyle(element);
    if (style.font) {
        return style.font;
    }
    var isFontDefined = style.fontFamily !== '';
    if (!isFontDefined) {
        return '';
    }
    var fontVariant = allowedVariants.includes(style.fontVariant) ? style.fontVariant : 'normal';
    return "".concat(style.fontStyle, " ").concat(fontVariant, " ").concat(style.fontWeight, " ").concat(style.fontSize, " / ").concat(style.lineHeight, " ").concat(style.fontFamily);
}
var cachedCanvas;
/**
 * Measures text width given text and font CSS shorthand.
 *
 * @param {string} text Text to measure
 * @param {string} font Font to use when measuring the text
 */
export function measureText(text, font) {
    var canvas = cachedCanvas || (cachedCanvas = document.createElement('canvas'));
    var context = canvas.getContext('2d');
    // Context type not supported
    if (!context) {
        return null;
    }
    context.font = font;
    var width = context.measureText(text).width;
    return Math.ceil(width);
}
/**
 * Updates input element width to fit its content given input element
 * @param {HTMLInputElement} element
 */
export function updateInputWidth(element) {
    if (typeof document === 'undefined' || !element) {
        return null;
    }
    var font = getFontShorthand(element);
    var text = element.value || element.placeholder;
    var width = measureText(text, font);
    if (width === null) {
        return null;
    }
    element.style.width = "".concat(width, "px");
    return width;
}
export default updateInputWidth;
